/**
 * Input Validation & Sanitization
 */

/**
 * Validate email format
 */
function isValidEmail(email) {
  if (!email || typeof email !== 'string') return false;
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  return emailRegex.test(email) && email.length <= 255;
}

/**
 * Sanitize string - remove dangerous characters
 */
function sanitizeString(str, maxLength = 1000) {
  if (!str || typeof str !== 'string') return '';
  return str
    .trim()
    .substring(0, maxLength)
    .replace(/[<>]/g, ''); // Basic XSS prevention
}

/**
 * Sanitize email
 */
function sanitizeEmail(email) {
  if (!email || typeof email !== 'string') return null;
  return email.trim().toLowerCase().substring(0, 255);
}

/**
 * Validate UUID format
 */
function isValidUUID(uuid) {
  if (!uuid || typeof uuid !== 'string') return false;
  const uuidRegex = /^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i;
  return uuidRegex.test(uuid);
}

/**
 * Validate slug format
 */
function isValidSlug(slug) {
  if (!slug || typeof slug !== 'string') return false;
  const slugRegex = /^[a-z0-9-]+$/;
  return slugRegex.test(slug) && slug.length <= 100;
}

/**
 * Sanitize object - remove any __proto__ or constructor pollution attempts
 */
function sanitizeObject(obj) {
  if (!obj || typeof obj !== 'object') return {};
  const clean = {};
  for (const key of Object.keys(obj)) {
    if (key === '__proto__' || key === 'constructor' || key === 'prototype') continue;
    if (typeof obj[key] === 'string') {
      clean[key] = sanitizeString(obj[key]);
    } else if (typeof obj[key] === 'number') {
      clean[key] = obj[key];
    } else if (typeof obj[key] === 'boolean') {
      clean[key] = obj[key];
    }
    // Skip other types for safety
  }
  return clean;
}

module.exports = {
  isValidEmail,
  sanitizeString,
  sanitizeEmail,
  isValidUUID,
  isValidSlug,
  sanitizeObject
};
