/**
 * Quiz Logic Engine
 * Evaluates conditional rules to determine quiz flow and results
 */

/**
 * Evaluate a single condition
 * @param {Object} condition - {questionId, operator, value}
 * @param {Object} answers - {questionId: [optionIds] or textValue}
 * @returns {boolean}
 */
function evaluateCondition(condition, answers) {
  const { questionId, operator, value } = condition;
  const answer = answers[questionId];
  
  if (!answer && operator !== 'is_empty') {
    return false;
  }

  // Handle array answers (single/multiple choice)
  const answerArray = Array.isArray(answer) ? answer : [answer];
  const valueArray = Array.isArray(value) ? value : [value];

  switch (operator) {
    case 'equals':
    case 'is':
      // Check if answer exactly matches value
      if (Array.isArray(answer)) {
        return answerArray.length === valueArray.length && 
               answerArray.every(a => valueArray.includes(a));
      }
      return answer === value;

    case 'not_equals':
    case 'is_not':
      if (Array.isArray(answer)) {
        return !(answerArray.length === valueArray.length && 
                 answerArray.every(a => valueArray.includes(a)));
      }
      return answer !== value;

    case 'contains':
    case 'includes':
      // Check if any of the values are in the answer
      return valueArray.some(v => answerArray.includes(v));

    case 'not_contains':
    case 'excludes':
      // Check if none of the values are in the answer
      return !valueArray.some(v => answerArray.includes(v));

    case 'is_empty':
      return !answer || (Array.isArray(answer) && answer.length === 0);

    case 'is_not_empty':
      return answer && (!Array.isArray(answer) || answer.length > 0);

    case 'greater_than':
      return parseFloat(answer) > parseFloat(value);

    case 'less_than':
      return parseFloat(answer) < parseFloat(value);

    case 'greater_or_equal':
      return parseFloat(answer) >= parseFloat(value);

    case 'less_or_equal':
      return parseFloat(answer) <= parseFloat(value);

    case 'between':
      const num = parseFloat(answer);
      return num >= parseFloat(valueArray[0]) && num <= parseFloat(valueArray[1]);

    case 'starts_with':
      return String(answer).toLowerCase().startsWith(String(value).toLowerCase());

    case 'ends_with':
      return String(answer).toLowerCase().endsWith(String(value).toLowerCase());

    case 'matches_regex':
      try {
        const regex = new RegExp(value, 'i');
        return regex.test(String(answer));
      } catch {
        return false;
      }

    default:
      console.warn(`Unknown operator: ${operator}`);
      return false;
  }
}

/**
 * Evaluate all conditions in a rule
 * @param {Object} rule - Logic rule with conditions and match type
 * @param {Object} answers - User's answers
 * @returns {boolean}
 */
function evaluateRule(rule, answers) {
  const { conditions, condition_match = 'all' } = rule;
  
  if (!conditions || conditions.length === 0) {
    return true; // No conditions = always true
  }

  if (condition_match === 'all') {
    return conditions.every(c => evaluateCondition(c, answers));
  } else if (condition_match === 'any') {
    return conditions.some(c => evaluateCondition(c, answers));
  }
  
  return false;
}

/**
 * Find the result page for given answers
 * @param {Object} answers - User's answers
 * @param {Array} rules - All logic rules for the quiz
 * @param {Array} resultPages - All result pages with settings
 * @param {number} score - Calculated score (optional)
 * @returns {Object} Result page object
 */
function determineResult(answers, rules, resultPages, score = null) {
  // First check if we have score-based routing
  if (score !== null) {
    // Find result page based on score range
    for (const page of resultPages) {
      const settings = page.settings ? (typeof page.settings === 'string' ? JSON.parse(page.settings) : page.settings) : {};
      const minScore = settings.minScore !== undefined ? settings.minScore : -Infinity;
      const maxScore = settings.maxScore !== undefined ? settings.maxScore : Infinity;
      
      if (score >= minScore && score <= maxScore) {
        return page;
      }
    }
  }

  // Then check logic rules
  if (rules && rules.length > 0) {
    const sortedRules = [...rules]
      .filter(r => r.enabled && r.action_type === 'show_result')
      .sort((a, b) => (b.priority || 0) - (a.priority || 0));

    for (const rule of sortedRules) {
      if (evaluateRule(rule, answers)) {
        const targetPageId = rule.action_value?.resultPageId;
        const targetPage = resultPages.find(p => p.id === targetPageId);
        if (targetPage) return targetPage;
      }
    }
  }

  // Return default page
  const defaultPage = resultPages.find(p => p.is_default) || resultPages[0];
  return defaultPage || { id: null, title: 'Your Results', blocks: '[]' };
}

/**
 * Determine next question based on skip logic
 * @param {Array} rules - All logic rules
 * @param {Object} answers - Current answers
 * @param {number} currentQuestionId - Current question
 * @param {Array} questions - All questions in order
 * @returns {number|null} Next question ID or null if end
 */
function getNextQuestion(rules, answers, currentQuestionId, questions) {
  const skipRules = rules
    .filter(r => r.enabled && r.action_type === 'skip_to_question')
    .sort((a, b) => (b.priority || 0) - (a.priority || 0));

  for (const rule of skipRules) {
    if (evaluateRule(rule, answers)) {
      return rule.action_value?.questionId || null;
    }
  }

  // No skip rule matched - go to next question in sequence
  const currentIndex = questions.findIndex(q => q.id === currentQuestionId);
  if (currentIndex >= 0 && currentIndex < questions.length - 1) {
    return questions[currentIndex + 1].id;
  }

  return null; // End of quiz
}

/**
 * Calculate accumulated tags from answers
 * @param {Array} options - All selected options with their tags
 * @param {Array} rules - Tag rules
 * @param {Object} answers - User's answers
 * @returns {Array} Accumulated tags
 */
function calculateTags(options, rules, answers) {
  const tags = new Set();

  // Add tags from selected options
  for (const option of options) {
    if (option.tags && Array.isArray(option.tags)) {
      option.tags.forEach(tag => tags.add(tag));
    }
  }

  // Add tags from rules
  const tagRules = rules.filter(r => r.enabled && r.action_type === 'add_tags');
  for (const rule of tagRules) {
    if (evaluateRule(rule, answers)) {
      const ruleTags = rule.action_value?.tags || [];
      ruleTags.forEach(tag => tags.add(tag));
    }
  }

  return Array.from(tags);
}

/**
 * Calculate total score from answers
 * @param {Array} options - All selected options with scores
 * @param {Array} rules - Score rules
 * @param {Object} answers - User's answers
 * @returns {number} Total score
 */
function calculateScore(options, rules, answers) {
  let score = 0;

  // Add scores from selected options
  for (const option of options) {
    score += option.score || 0;
  }

  // Apply score rules
  const scoreRules = rules.filter(r => r.enabled && r.action_type === 'add_score');
  for (const rule of scoreRules) {
    if (evaluateRule(rule, answers)) {
      score += rule.action_value?.score || 0;
    }
  }

  return score;
}

module.exports = {
  evaluateCondition,
  evaluateRule,
  determineResult,
  getNextQuestion,
  calculateTags,
  calculateScore
};
