/**
 * Encryption Utility
 * For securely storing API keys in database
 */

const CryptoJS = require('crypto-js');

const ENCRYPTION_KEY = process.env.ENCRYPTION_KEY || 'default-key-change-in-production';

/**
 * Encrypt a string
 */
function encrypt(text) {
  if (!text) return null;
  return CryptoJS.AES.encrypt(text, ENCRYPTION_KEY).toString();
}

/**
 * Decrypt a string
 */
function decrypt(ciphertext) {
  if (!ciphertext) return null;
  try {
    const bytes = CryptoJS.AES.decrypt(ciphertext, ENCRYPTION_KEY);
    return bytes.toString(CryptoJS.enc.Utf8);
  } catch (error) {
    console.error('Decryption error:', error);
    return null;
  }
}

/**
 * Encrypt an object (JSON)
 */
function encryptObject(obj) {
  if (!obj) return null;
  return encrypt(JSON.stringify(obj));
}

/**
 * Decrypt to an object (JSON)
 */
function decryptObject(ciphertext) {
  if (!ciphertext) return null;
  try {
    const decrypted = decrypt(ciphertext);
    return JSON.parse(decrypted);
  } catch (error) {
    console.error('Decryption/parse error:', error);
    return null;
  }
}

module.exports = { encrypt, decrypt, encryptObject, decryptObject };
