#!/usr/bin/env node
/**
 * QuizFunnel Setup Script
 * Creates the admin user for new installations
 * Run: node setup.js
 */

require('dotenv').config();
const bcrypt = require('bcryptjs');
const readline = require('readline');

const rl = readline.createInterface({
  input: process.stdin,
  output: process.stdout
});

function question(prompt) {
  return new Promise(resolve => rl.question(prompt, resolve));
}

async function setup() {
  console.log('\n🚀 QuizFunnel Setup\n');
  console.log('This will create your admin account.\n');

  // Check database
  try {
    const { pool } = require('./utils/db');
    await pool.execute('SELECT 1');
    console.log('✅ Database connected\n');
  } catch (error) {
    console.error('❌ Database connection failed:', error.message);
    console.log('\nCheck your .env file and make sure the database exists.\n');
    process.exit(1);
  }

  // Check if admin exists
  const { pool } = require('./utils/db');
  const [existing] = await pool.execute('SELECT id FROM users WHERE role = "admin"');
  
  if (existing.length > 0) {
    const proceed = await question('An admin already exists. Create another? (y/n): ');
    if (proceed.toLowerCase() !== 'y') {
      console.log('\nSetup cancelled.\n');
      process.exit(0);
    }
  }

  // Get admin details
  const email = await question('Admin Email: ');
  const password = await question('Admin Password: ');
  const name = await question('Name (optional): ') || 'Admin';

  if (!email || !password) {
    console.log('\n❌ Email and password required.\n');
    process.exit(1);
  }

  if (password.length < 8) {
    console.log('\n❌ Password must be at least 8 characters.\n');
    process.exit(1);
  }

  // Check if email exists
  const [emailExists] = await pool.execute('SELECT id FROM users WHERE email = ?', [email.toLowerCase()]);
  if (emailExists.length > 0) {
    console.log('\n❌ Email already in use.\n');
    process.exit(1);
  }

  // Create admin
  const hashedPassword = await bcrypt.hash(password, 10);
  
  await pool.execute(
    'INSERT INTO users (email, password, name, role) VALUES (?, ?, ?, ?)',
    [email.toLowerCase(), hashedPassword, name, 'admin']
  );

  console.log('\n✅ Admin created successfully!\n');
  console.log('Login at: /admin\n');
  console.log(`Email: ${email}`);
  console.log(`Password: ${password}\n`);

  rl.close();
  process.exit(0);
}

setup().catch(err => {
  console.error('Setup failed:', err);
  process.exit(1);
});
