/**
 * Integration Sync Service
 * Orchestrates syncing leads to all enabled integrations
 * Now reads from settings table for API keys and auto-tags
 */

const { pool } = require('../utils/db');
const axios = require('axios');

/**
 * Get setting value from database
 */
async function getSetting(key) {
  const [rows] = await pool.execute('SELECT value FROM settings WHERE key_name = ?', [key]);
  return rows[0]?.value || null;
}

/**
 * Get multiple settings
 */
async function getSettings(keys) {
  const placeholders = keys.map(() => '?').join(',');
  const [rows] = await pool.execute(
    `SELECT key_name, value FROM settings WHERE key_name IN (${placeholders})`,
    keys
  );
  const result = {};
  rows.forEach(r => { result[r.key_name] = r.value; });
  return result;
}

/**
 * Sync lead to all enabled integrations
 * Called after quiz completion
 */
async function syncLeadToAllIntegrations(leadId) {
  const results = {
    mailchimp: null,
    kit: null,
    webhook: null
  };

  // Run all syncs in parallel
  const tasks = [
    syncLeadToMailchimp(leadId)
      .then(r => { results.mailchimp = r ? 'success' : 'skipped'; })
      .catch(e => { results.mailchimp = `error: ${e.message}`; }),
    
    syncLeadToKit(leadId)
      .then(r => { results.kit = r ? 'success' : 'skipped'; })
      .catch(e => { results.kit = `error: ${e.message}`; }),
    
    syncLeadToWebhooks(leadId)
      .then(r => { results.webhook = r ? 'success' : 'skipped'; })
      .catch(e => { results.webhook = `error: ${e.message}`; })
  ];

  await Promise.all(tasks);
  console.log(`Lead ${leadId} sync results:`, results);
  return results;
}

/**
 * Sync lead to Mailchimp using settings
 */
async function syncLeadToMailchimp(leadId) {
  const settings = await getSettings(['mailchimp_api_key', 'mailchimp_list_id', 'mailchimp_tag_on_signup']);
  
  if (!settings.mailchimp_api_key || !settings.mailchimp_list_id) {
    return null; // Not configured
  }

  // Get lead data
  const [leads] = await pool.execute(`
    SELECT l.*, q.title as quiz_title, p.title as result_title, p.mailchimp_tag as result_mailchimp_tag
    FROM leads l
    JOIN quizzes q ON l.quiz_id = q.id
    LEFT JOIN pages p ON l.result_page_id = p.id
    WHERE l.id = ?
  `, [leadId]);

  if (leads.length === 0 || !leads[0].email) return null;

  const lead = leads[0];
  const apiKey = settings.mailchimp_api_key;
  const listId = settings.mailchimp_list_id;
  const dc = apiKey.split('-').pop();

  // Build tags array
  const tags = [];
  if (settings.mailchimp_tag_on_signup) {
    tags.push(settings.mailchimp_tag_on_signup);
  }
  if (lead.result_mailchimp_tag) {
    tags.push(lead.result_mailchimp_tag);
  }
  if (lead.tags) {
    try {
      const leadTags = JSON.parse(lead.tags);
      tags.push(...leadTags);
    } catch (e) {}
  }

  // Hash email for Mailchimp
  const crypto = require('crypto');
  const emailHash = crypto.createHash('md5').update(lead.email.toLowerCase()).digest('hex');

  // Parse name
  const nameParts = (lead.name || '').split(' ');
  const firstName = nameParts[0] || '';
  const lastName = nameParts.slice(1).join(' ') || '';

  try {
    // Add/update subscriber
    await axios.put(
      `https://${dc}.api.mailchimp.com/3.0/lists/${listId}/members/${emailHash}`,
      {
        email_address: lead.email.toLowerCase(),
        status_if_new: 'subscribed',
        merge_fields: {
          FNAME: firstName,
          LNAME: lastName,
          PHONE: lead.phone || ''
        }
      },
      { headers: { 'Authorization': `Bearer ${apiKey}` } }
    );

    // Add tags if any
    if (tags.length > 0) {
      await axios.post(
        `https://${dc}.api.mailchimp.com/3.0/lists/${listId}/members/${emailHash}/tags`,
        { tags: tags.map(t => ({ name: t, status: 'active' })) },
        { headers: { 'Authorization': `Bearer ${apiKey}` } }
      );
    }

    // Mark as synced
    await pool.execute('UPDATE leads SET synced_mailchimp = 1 WHERE id = ?', [leadId]);
    return true;
  } catch (error) {
    console.error('Mailchimp sync error:', error.response?.data || error.message);
    throw error;
  }
}

/**
 * Sync lead to Kit (ConvertKit) using settings
 */
async function syncLeadToKit(leadId) {
  const settings = await getSettings(['kit_api_key', 'kit_form_id', 'kit_tag_on_signup']);
  
  if (!settings.kit_api_key || !settings.kit_form_id) {
    return null; // Not configured
  }

  // Get lead data
  const [leads] = await pool.execute(`
    SELECT l.*, p.kit_tag as result_kit_tag
    FROM leads l
    LEFT JOIN pages p ON l.result_page_id = p.id
    WHERE l.id = ?
  `, [leadId]);

  if (leads.length === 0 || !leads[0].email) return null;

  const lead = leads[0];
  const apiKey = settings.kit_api_key;
  const formId = settings.kit_form_id;

  // Build tags array
  const tags = [];
  if (settings.kit_tag_on_signup) {
    tags.push(settings.kit_tag_on_signup);
  }
  if (lead.result_kit_tag) {
    tags.push(lead.result_kit_tag);
  }

  // Parse name
  const nameParts = (lead.name || '').split(' ');
  const firstName = nameParts[0] || '';

  try {
    // Subscribe to form
    await axios.post(
      `https://api.convertkit.com/v3/forms/${formId}/subscribe`,
      {
        api_secret: apiKey,
        email: lead.email.toLowerCase(),
        first_name: firstName,
        tags: tags
      }
    );

    // Mark as synced
    await pool.execute('UPDATE leads SET synced_kit = 1 WHERE id = ?', [leadId]);
    return true;
  } catch (error) {
    console.error('Kit sync error:', error.response?.data || error.message);
    throw error;
  }
}

/**
 * Sync lead to webhooks (Zapier, Pabbly) with separate new_lead and quiz_complete hooks
 */
async function syncLeadToWebhooks(leadId, eventType = 'quiz_complete') {
  const settings = await getSettings([
    'zapier_webhook_url', 'pabbly_webhook_url',
    'zapier_webhook_new_lead', 'zapier_webhook_quiz_complete',
    'pabbly_webhook_new_lead', 'pabbly_webhook_quiz_complete'
  ]);
  
  // Determine which webhooks to call based on event type
  let webhookUrls = [];
  
  if (eventType === 'new_lead') {
    webhookUrls = [
      settings.zapier_webhook_new_lead,
      settings.pabbly_webhook_new_lead
    ].filter(Boolean);
  } else if (eventType === 'quiz_complete') {
    webhookUrls = [
      settings.zapier_webhook_url,
      settings.pabbly_webhook_url,
      settings.zapier_webhook_quiz_complete,
      settings.pabbly_webhook_quiz_complete
    ].filter(Boolean);
  }

  // Remove duplicates
  webhookUrls = [...new Set(webhookUrls)];

  if (webhookUrls.length === 0) {
    return null; // No webhooks configured
  }

  // Get lead data
  const [leads] = await pool.execute(`
    SELECT l.*, q.title as quiz_title, q.slug as quiz_slug, p.title as result_title
    FROM leads l
    JOIN quizzes q ON l.quiz_id = q.id
    LEFT JOIN pages p ON l.result_page_id = p.id
    WHERE l.id = ?
  `, [leadId]);

  if (leads.length === 0) return null;

  const lead = leads[0];

  const payload = {
    event: eventType === 'new_lead' ? 'lead.captured' : 'quiz.completed',
    timestamp: new Date().toISOString(),
    lead: {
      id: lead.uuid,
      email: lead.email,
      name: lead.name,
      phone: lead.phone,
      custom_fields: lead.custom_fields ? JSON.parse(lead.custom_fields) : {},
      tags: lead.tags ? JSON.parse(lead.tags) : [],
      score: lead.score
    },
    quiz: {
      title: lead.quiz_title,
      slug: lead.quiz_slug
    },
    result: {
      title: lead.result_title
    },
    tracking: {
      utm_source: lead.utm_source,
      utm_medium: lead.utm_medium,
      utm_campaign: lead.utm_campaign,
      referrer: lead.referrer
    }
  };

  // Send to all webhooks
  for (const url of webhookUrls) {
    try {
      await axios.post(url, payload, {
        headers: { 'Content-Type': 'application/json' },
        timeout: 30000
      });
    } catch (error) {
      console.error(`Webhook error for ${url}:`, error.message);
    }
  }

  // Mark as synced
  await pool.execute('UPDATE leads SET synced_webhook = 1 WHERE id = ?', [leadId]);
  return true;
}

module.exports = {
  syncLeadToAllIntegrations,
  syncLeadToMailchimp,
  syncLeadToKit,
  syncLeadToWebhooks,
  getSetting,
  getSettings
};
