/**
 * Kit (ConvertKit) Integration Service
 * Sync leads to Kit forms with tags
 */

const axios = require('axios');
const { pool } = require('../utils/db');
const { decryptObject } = require('../utils/encryption');

const KIT_API_URL = 'https://api.convertkit.com/v3';

class KitService {
  constructor(apiKey, apiSecret = null) {
    this.apiKey = apiKey;
    this.apiSecret = apiSecret || apiKey;
  }

  /**
   * Make authenticated request to Kit API
   */
  async request(method, endpoint, data = null) {
    try {
      const params = method === 'GET' ? { api_key: this.apiKey } : {};
      const body = method !== 'GET' ? { 
        ...data, 
        api_secret: this.apiSecret 
      } : undefined;

      const response = await axios({
        method,
        url: `${KIT_API_URL}${endpoint}`,
        params,
        data: body,
        headers: {
          'Content-Type': 'application/json'
        }
      });
      return response.data;
    } catch (error) {
      console.error('Kit API error:', error.response?.data || error.message);
      throw new Error(error.response?.data?.message || 'Kit API error');
    }
  }

  /**
   * Get all forms
   */
  async getForms() {
    const data = await this.request('GET', '/forms');
    return data.forms.map(form => ({
      id: form.id,
      name: form.name,
      type: form.type,
      subscriberCount: form.total_subscriptions
    }));
  }

  /**
   * Get all tags
   */
  async getTags() {
    const data = await this.request('GET', '/tags');
    return data.tags.map(tag => ({
      id: tag.id,
      name: tag.name
    }));
  }

  /**
   * Add subscriber to a form
   */
  async addSubscriberToForm(formId, subscriber, tags = []) {
    const { email, firstName, lastName, customFields } = subscriber;

    const fields = {
      first_name: firstName || '',
      ...customFields
    };

    // If we have a last name, add it as custom field
    if (lastName) {
      fields.last_name = lastName;
    }

    const data = await this.request('POST', `/forms/${formId}/subscribe`, {
      email: email.toLowerCase(),
      first_name: firstName || '',
      fields,
      tags: tags.map(t => typeof t === 'number' ? t : t.toString())
    });

    return data.subscription;
  }

  /**
   * Add tag to subscriber
   */
  async addTagToSubscriber(tagId, email) {
    const data = await this.request('POST', `/tags/${tagId}/subscribe`, {
      email: email.toLowerCase()
    });
    return data.subscription;
  }

  /**
   * Create a new tag
   */
  async createTag(tagName) {
    const data = await this.request('POST', '/tags', {
      tag: { name: tagName }
    });
    return data.tag;
  }

  /**
   * Get subscriber by email
   */
  async getSubscriber(email) {
    try {
      const data = await this.request('GET', `/subscribers?email_address=${encodeURIComponent(email)}`);
      return data.subscribers[0] || null;
    } catch (error) {
      return null;
    }
  }
}

/**
 * Sync lead to Kit
 * Called after quiz completion
 */
async function syncLeadToKit(leadId) {
  try {
    // Get lead with integration settings
    const [leads] = await pool.execute(
      `SELECT l.*, qi.tag_mapping, qi.field_mapping, i.credentials, i.settings, i.id as integration_id
       FROM leads l
       JOIN quiz_integrations qi ON l.quiz_id = qi.quiz_id
       JOIN integrations i ON qi.integration_id = i.id
       WHERE l.id = ? AND i.platform = 'kit' AND i.enabled = 1`,
      [leadId]
    );

    if (leads.length === 0) {
      return null; // No Kit integration for this quiz
    }

    const lead = leads[0];
    const creds = decryptObject(lead.credentials);
    const settings = JSON.parse(lead.settings || '{}');
    const tagMapping = JSON.parse(lead.tag_mapping || '{}');

    if (!creds?.apiKey || !settings.formId) {
      console.log('Kit not configured properly');
      return null;
    }

    const kit = new KitService(creds.apiKey, creds.apiSecret);

    // Parse lead name
    const nameParts = (lead.name || '').split(' ');
    const firstName = nameParts[0] || '';
    const lastName = nameParts.slice(1).join(' ') || '';

    // Get tags from lead
    let tags = [];
    if (lead.tags) {
      const leadTags = JSON.parse(lead.tags);
      tags = [...leadTags];
    }

    // Add mapped tags based on result
    if (lead.result_page_id && tagMapping[lead.result_page_id]) {
      tags.push(tagMapping[lead.result_page_id]);
    }

    // Sync to Kit
    await kit.addSubscriberToForm(settings.formId, {
      email: lead.email,
      firstName,
      lastName,
      customFields: JSON.parse(lead.custom_fields || '{}')
    }, tags);

    // Mark as synced
    await pool.execute(
      'UPDATE leads SET synced_kit = 1 WHERE id = ?',
      [leadId]
    );

    // Update integration last sync time
    await pool.execute(
      'UPDATE integrations SET last_sync = NOW(), last_error = NULL WHERE id = ?',
      [lead.integration_id]
    );

    return true;
  } catch (error) {
    console.error('Kit sync error:', error);
    
    // Log error to integration
    await pool.execute(
      'UPDATE integrations SET last_error = ? WHERE platform = "kit"',
      [error.message]
    );
    
    throw error;
  }
}

module.exports = {
  KitService,
  syncLeadToKit
};
