/**
 * QuizFunnel - Single Tenant Edition
 * Deploy per client on their own hosting
 */

require('dotenv').config();
const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const compression = require('compression');
const path = require('path');
const rateLimit = require('express-rate-limit');

// Import routes
const authRoutes = require('./routes/auth');
const quizRoutes = require('./routes/quiz');
const adminRoutes = require('./routes/admin');
const leadRoutes = require('./routes/leads');
const integrationRoutes = require('./routes/integrations');
const uploadRoutes = require('./routes/upload');
const settingsRoutes = require('./routes/settings');

// Import middleware
const { errorHandler } = require('./middleware/errorHandler');
const domainDetect = require('./middleware/domainDetect');

const app = express();
const PORT = process.env.PORT || 3000;

// Trust proxy
app.set('trust proxy', 1);

// Security
app.use(helmet({
  contentSecurityPolicy: {
    directives: {
      defaultSrc: ["'self'"],
      styleSrc: ["'self'", "'unsafe-inline'", "https://fonts.googleapis.com", "https://cdnjs.cloudflare.com"],
      fontSrc: ["'self'", "https://fonts.gstatic.com", "https://cdnjs.cloudflare.com"],
      scriptSrc: ["'self'", "'unsafe-inline'", "'unsafe-eval'", "https://unpkg.com", "https://cdnjs.cloudflare.com", "https://cdn.jsdelivr.net", "https://cdn.tailwindcss.com"],
      imgSrc: ["'self'", "data:", "https:", "blob:"],
      connectSrc: ["'self'", "https://api.mailchimp.com", "https://api.convertkit.com"],
      frameSrc: ["'self'", "https://www.youtube.com", "https://player.vimeo.com", "https://www.loom.com"],
    },
  },
  crossOriginEmbedderPolicy: false,
}));

// CORS
app.use(cors({ origin: true, credentials: true }));

// Compression
app.use(compression());

// Body parsing
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Rate limiting
app.use('/api/', rateLimit({
  windowMs: 15 * 60 * 1000,
  max: 500,
  message: { error: 'Too many requests' }
}));

app.use('/api/auth/login', rateLimit({
  windowMs: 15 * 60 * 1000,
  max: 10,
  message: { error: 'Too many login attempts' }
}));

// Static files
app.use(express.static(path.join(__dirname, 'public_html')));
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));

// Domain detection - runs on every request
app.use(domainDetect);

// API Routes
app.use('/api/auth', authRoutes);
app.use('/api/quiz', quizRoutes);
app.use('/api/admin', adminRoutes);
app.use('/api/leads', leadRoutes);
app.use('/api/integrations', integrationRoutes);
app.use('/api/upload', uploadRoutes);
app.use('/api/settings', settingsRoutes);

// Health check
app.get('/api/health', (req, res) => {
  res.json({ status: 'ok', timestamp: new Date().toISOString() });
});

// Quiz pages - clean URLs
app.get('/quiz/:quizSlug', (req, res) => {
  res.sendFile(path.join(__dirname, 'public_html/quiz/index.html'));
});

app.get('/q/:quizSlug', (req, res) => {
  res.sendFile(path.join(__dirname, 'public_html/quiz/index.html'));
});

// Admin dashboard
app.get('/admin*', (req, res) => {
  res.sendFile(path.join(__dirname, 'public_html/admin/index.html'));
});

// Catch-all
app.get('*', (req, res) => {
  if (req.path.startsWith('/api/')) {
    return res.status(404).json({ error: 'Endpoint not found' });
  }
  
  // If a custom domain quiz was detected, serve quiz page directly
  if (req.quiz && req.path === '/') {
    return res.sendFile(path.join(__dirname, 'public_html/quiz/index.html'));
  }
  
  res.sendFile(path.join(__dirname, 'public_html/index.html'));
});

// Error handling
app.use(errorHandler);

// Start server
app.listen(PORT, () => {
  console.log(`\n🚀 QuizFunnel running on port ${PORT}`);
  console.log(`   Dashboard: ${process.env.APP_URL || `http://localhost:${PORT}`}/admin`);
  console.log(`   Quizzes: ${process.env.APP_URL || `http://localhost:${PORT}`}/quiz/your-quiz-slug\n`);
});

module.exports = app;
