/**
 * Leads Routes - Single Tenant Version
 * Lead management, export, stats
 */

const express = require('express');
const router = express.Router();
const { pool } = require('../utils/db');
const { authenticate } = require('../middleware/auth');
const { asyncHandler } = require('../middleware/errorHandler');

// All routes require authentication
router.use(authenticate);

/**
 * GET /api/leads
 * List leads with filtering and pagination
 */
router.get('/', asyncHandler(async (req, res) => {
  const { quizId, search, tag, completed, page = 1, limit = 50, sort = 'created_at', order = 'DESC' } = req.query;

  const conditions = [];
  const values = [];

  if (quizId) {
    conditions.push('l.quiz_id = ?');
    values.push(quizId);
  }

  if (search) {
    conditions.push('(l.email LIKE ? OR l.name LIKE ?)');
    values.push(`%${search}%`, `%${search}%`);
  }

  if (tag) {
    conditions.push('JSON_CONTAINS(l.tags, ?)');
    values.push(`"${tag}"`);
  }

  if (completed !== undefined) {
    conditions.push('l.completed_quiz = ?');
    values.push(completed === 'true' ? 1 : 0);
  }

  const whereClause = conditions.length > 0 ? `WHERE ${conditions.join(' AND ')}` : '';

  // Count total
  const [countResult] = await pool.execute(
    `SELECT COUNT(*) as total FROM leads l ${whereClause}`,
    values
  );

  const offset = (parseInt(page) - 1) * parseInt(limit);
  const allowedSorts = ['created_at', 'email', 'name', 'score'];
  const sortColumn = allowedSorts.includes(sort) ? sort : 'created_at';
  const sortOrder = order.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';

  // Get leads
  const [leads] = await pool.execute(
    `SELECT l.*, q.title as quiz_title, p.title as result_title
     FROM leads l
     LEFT JOIN quizzes q ON l.quiz_id = q.id
     LEFT JOIN pages p ON l.result_page_id = p.id
     ${whereClause}
     ORDER BY l.${sortColumn} ${sortOrder}
     LIMIT ? OFFSET ?`,
    [...values, parseInt(limit), offset]
  );

  res.json({
    leads,
    pagination: {
      total: countResult[0].total,
      page: parseInt(page),
      limit: parseInt(limit),
      pages: Math.ceil(countResult[0].total / parseInt(limit))
    }
  });
}));

/**
 * GET /api/leads/stats
 * Get lead statistics
 */
router.get('/stats', asyncHandler(async (req, res) => {
  const { quizId, days = 30 } = req.query;

  let quizFilter = '';
  const values = [];

  if (quizId) {
    quizFilter = 'AND quiz_id = ?';
    values.push(quizId);
  }

  // Total counts
  const [totals] = await pool.execute(`
    SELECT 
      COUNT(*) as total,
      SUM(CASE WHEN completed_quiz = 1 THEN 1 ELSE 0 END) as completed,
      SUM(CASE WHEN completed_quiz = 0 THEN 1 ELSE 0 END) as partial
    FROM leads
    WHERE 1=1 ${quizFilter}
  `, values);

  // Leads per day
  const [daily] = await pool.execute(`
    SELECT DATE(created_at) as date, COUNT(*) as count
    FROM leads
    WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY) ${quizFilter}
    GROUP BY DATE(created_at)
    ORDER BY date
  `, [parseInt(days), ...(quizId ? [quizId] : [])]);

  // Top tags
  const [tags] = await pool.execute(`
    SELECT tag, COUNT(*) as count
    FROM leads, JSON_TABLE(tags, '$[*]' COLUMNS(tag VARCHAR(100) PATH '$')) as jt
    WHERE tags IS NOT NULL ${quizFilter}
    GROUP BY tag
    ORDER BY count DESC
    LIMIT 10
  `, values);

  // Completion rate
  const completionRate = totals[0].total > 0 
    ? (totals[0].completed / totals[0].total * 100).toFixed(1)
    : 0;

  res.json({
    totals: totals[0],
    completionRate,
    daily,
    topTags: tags
  });
}));

/**
 * GET /api/leads/:leadId
 * Get single lead details
 */
router.get('/:leadId', asyncHandler(async (req, res) => {
  const { leadId } = req.params;

  const [leads] = await pool.execute(`
    SELECT l.*, q.title as quiz_title, p.title as result_title
    FROM leads l
    LEFT JOIN quizzes q ON l.quiz_id = q.id
    LEFT JOIN pages p ON l.result_page_id = p.id
    WHERE l.id = ?
  `, [leadId]);

  if (leads.length === 0) {
    return res.status(404).json({ error: 'Lead not found' });
  }

  const lead = leads[0];

  // Get detailed answers with question text
  const [answers] = await pool.execute(`
    SELECT q.question_text, q.question_type, qo.option_text, la.text_value
    FROM lead_answers la
    JOIN questions q ON la.question_id = q.id
    LEFT JOIN question_options qo ON JSON_CONTAINS(la.option_ids, CAST(qo.id as JSON))
    WHERE la.lead_id = ?
    ORDER BY q.order_index
  `, [leadId]);

  res.json({ lead, answers });
}));

/**
 * DELETE /api/leads/:leadId
 * Delete lead
 */
router.delete('/:leadId', asyncHandler(async (req, res) => {
  const { leadId } = req.params;
  await pool.execute('DELETE FROM leads WHERE id = ?', [leadId]);
  res.json({ message: 'Lead deleted' });
}));

/**
 * DELETE /api/leads
 * Bulk delete leads
 */
router.delete('/', asyncHandler(async (req, res) => {
  const { leadIds } = req.body;

  if (!Array.isArray(leadIds) || leadIds.length === 0) {
    return res.status(400).json({ error: 'Lead IDs array required' });
  }

  const placeholders = leadIds.map(() => '?').join(',');
  await pool.execute(`DELETE FROM leads WHERE id IN (${placeholders})`, leadIds);

  res.json({ message: `${leadIds.length} leads deleted` });
}));

/**
 * GET /api/leads/export/csv
 * Export leads as CSV
 */
router.get('/export/csv', asyncHandler(async (req, res) => {
  const { quizId, startDate, endDate } = req.query;

  const conditions = [];
  const values = [];

  if (quizId) {
    conditions.push('l.quiz_id = ?');
    values.push(quizId);
  }

  if (startDate) {
    conditions.push('l.created_at >= ?');
    values.push(startDate);
  }

  if (endDate) {
    conditions.push('l.created_at <= ?');
    values.push(endDate);
  }

  const whereClause = conditions.length > 0 ? `WHERE ${conditions.join(' AND ')}` : '';

  const [leads] = await pool.execute(`
    SELECT l.email, l.name, l.phone, l.custom_fields, l.tags, l.score, 
           l.completed_quiz, l.created_at, q.title as quiz_title, p.title as result_title
    FROM leads l
    LEFT JOIN quizzes q ON l.quiz_id = q.id
    LEFT JOIN pages p ON l.result_page_id = p.id
    ${whereClause}
    ORDER BY l.created_at DESC
  `, values);

  // Build CSV
  const headers = ['Email', 'Name', 'Phone', 'Quiz', 'Result', 'Score', 'Tags', 'Completed', 'Date'];
  const rows = leads.map(l => [
    l.email || '',
    l.name || '',
    l.phone || '',
    l.quiz_title || '',
    l.result_title || '',
    l.score || '',
    l.tags ? JSON.parse(l.tags).join('; ') : '',
    l.completed_quiz ? 'Yes' : 'No',
    new Date(l.created_at).toISOString()
  ]);

  const csv = [
    headers.join(','),
    ...rows.map(row => row.map(cell => `"${String(cell).replace(/"/g, '""')}"`).join(','))
  ].join('\n');

  res.setHeader('Content-Type', 'text/csv');
  res.setHeader('Content-Disposition', `attachment; filename="leads-${Date.now()}.csv"`);
  res.send(csv);
}));

/**
 * POST /api/leads/:leadId/resync
 * Retry integration sync for lead
 */
router.post('/:leadId/resync', asyncHandler(async (req, res) => {
  const { leadId } = req.params;

  // Reset sync flags
  await pool.execute(
    'UPDATE leads SET synced_mailchimp = 0, synced_kit = 0, synced_sheets = 0, synced_webhook = 0 WHERE id = ?',
    [leadId]
  );

  // Trigger sync
  try {
    const { syncLeadToAllIntegrations } = require('../services/sync');
    await syncLeadToAllIntegrations(leadId);
    res.json({ message: 'Sync triggered' });
  } catch (error) {
    res.status(500).json({ error: 'Sync failed', details: error.message });
  }
}));

module.exports = router;
