/**
 * Auth Routes - Single Tenant Version
 * Login, password reset, team management
 */

const express = require('express');
const router = express.Router();
const bcrypt = require('bcryptjs');
const crypto = require('crypto');
const { pool } = require('../utils/db');
const { generateTokens, authenticate, requireAdmin } = require('../middleware/auth');
const { asyncHandler } = require('../middleware/errorHandler');

/**
 * POST /api/auth/login
 * Login to dashboard
 */
router.post('/login', asyncHandler(async (req, res) => {
  const { email, password } = req.body;

  if (!email || !password) {
    return res.status(400).json({ error: 'Email and password required' });
  }

  const [users] = await pool.execute(
    'SELECT * FROM users WHERE email = ?',
    [email.toLowerCase()]
  );

  if (users.length === 0) {
    return res.status(401).json({ error: 'Invalid credentials' });
  }

  const user = users[0];
  const validPassword = await bcrypt.compare(password, user.password);

  if (!validPassword) {
    return res.status(401).json({ error: 'Invalid credentials' });
  }

  // Update last login
  await pool.execute('UPDATE users SET last_login = NOW() WHERE id = ?', [user.id]);

  const tokens = generateTokens({ userId: user.id });

  res.json({
    user: {
      id: user.id,
      email: user.email,
      name: user.name,
      role: user.role
    },
    ...tokens
  });
}));

/**
 * GET /api/auth/me
 * Get current user info
 */
router.get('/me', authenticate, asyncHandler(async (req, res) => {
  // Get counts for dashboard
  const [quizCount] = await pool.execute('SELECT COUNT(*) as count FROM quizzes');
  const [leadCount] = await pool.execute('SELECT COUNT(*) as count FROM leads');

  res.json({
    user: req.user,
    stats: {
      quizzes: quizCount[0].count,
      leads: leadCount[0].count
    }
  });
}));

/**
 * PUT /api/auth/me
 * Update current user
 */
router.put('/me', authenticate, asyncHandler(async (req, res) => {
  const { name, email, currentPassword, newPassword } = req.body;

  const updates = [];
  const values = [];

  if (name !== undefined) {
    updates.push('name = ?');
    values.push(name);
  }

  if (email !== undefined) {
    // Check if email is taken
    const [existing] = await pool.execute(
      'SELECT id FROM users WHERE email = ? AND id != ?',
      [email.toLowerCase(), req.userId]
    );
    if (existing.length > 0) {
      return res.status(409).json({ error: 'Email already in use' });
    }
    updates.push('email = ?');
    values.push(email.toLowerCase());
  }

  // Password change
  if (newPassword) {
    if (!currentPassword) {
      return res.status(400).json({ error: 'Current password required' });
    }

    const [users] = await pool.execute('SELECT password FROM users WHERE id = ?', [req.userId]);
    const validPassword = await bcrypt.compare(currentPassword, users[0].password);

    if (!validPassword) {
      return res.status(401).json({ error: 'Current password is incorrect' });
    }

    const hashedPassword = await bcrypt.hash(newPassword, 10);
    updates.push('password = ?');
    values.push(hashedPassword);
  }

  if (updates.length === 0) {
    return res.status(400).json({ error: 'No fields to update' });
  }

  values.push(req.userId);
  await pool.execute(`UPDATE users SET ${updates.join(', ')} WHERE id = ?`, values);

  res.json({ message: 'Profile updated' });
}));

/**
 * POST /api/auth/forgot-password
 * Request password reset
 */
router.post('/forgot-password', asyncHandler(async (req, res) => {
  const { email } = req.body;

  if (!email) {
    return res.status(400).json({ error: 'Email required' });
  }

  const [users] = await pool.execute('SELECT id FROM users WHERE email = ?', [email.toLowerCase()]);

  // Always return success to prevent email enumeration
  if (users.length === 0) {
    return res.json({ message: 'If an account exists, you will receive a reset email' });
  }

  // Generate reset token
  const resetToken = crypto.randomBytes(32).toString('hex');
  const expires = new Date(Date.now() + 3600000); // 1 hour

  await pool.execute(
    'UPDATE users SET reset_token = ?, reset_token_expires = ? WHERE id = ?',
    [resetToken, expires, users[0].id]
  );

  // TODO: Send email with reset link
  // For now, log it (in production, send email)
  console.log(`Password reset token for ${email}: ${resetToken}`);

  res.json({ message: 'If an account exists, you will receive a reset email' });
}));

/**
 * POST /api/auth/reset-password
 * Reset password with token
 */
router.post('/reset-password', asyncHandler(async (req, res) => {
  const { token, password } = req.body;

  if (!token || !password) {
    return res.status(400).json({ error: 'Token and password required' });
  }

  const [users] = await pool.execute(
    'SELECT id FROM users WHERE reset_token = ? AND reset_token_expires > NOW()',
    [token]
  );

  if (users.length === 0) {
    return res.status(400).json({ error: 'Invalid or expired reset token' });
  }

  const hashedPassword = await bcrypt.hash(password, 10);

  await pool.execute(
    'UPDATE users SET password = ?, reset_token = NULL, reset_token_expires = NULL WHERE id = ?',
    [hashedPassword, users[0].id]
  );

  res.json({ message: 'Password reset successfully' });
}));

// ============================================
// Team Management (Admin only)
// ============================================

/**
 * GET /api/auth/team
 * List team members
 */
router.get('/team', authenticate, requireAdmin, asyncHandler(async (req, res) => {
  const [users] = await pool.execute(
    'SELECT id, email, name, role, last_login, created_at FROM users ORDER BY created_at'
  );

  res.json({ users });
}));

/**
 * POST /api/auth/team
 * Invite team member
 */
router.post('/team', authenticate, requireAdmin, asyncHandler(async (req, res) => {
  const { email, name, role = 'editor', password } = req.body;

  if (!email || !password) {
    return res.status(400).json({ error: 'Email and password required' });
  }

  // Check if exists
  const [existing] = await pool.execute('SELECT id FROM users WHERE email = ?', [email.toLowerCase()]);
  if (existing.length > 0) {
    return res.status(409).json({ error: 'User already exists' });
  }

  const hashedPassword = await bcrypt.hash(password, 10);

  const [result] = await pool.execute(
    'INSERT INTO users (email, password, name, role) VALUES (?, ?, ?, ?)',
    [email.toLowerCase(), hashedPassword, name, role]
  );

  res.status(201).json({
    user: {
      id: result.insertId,
      email: email.toLowerCase(),
      name,
      role
    }
  });
}));

/**
 * PUT /api/auth/team/:userId
 * Update team member
 */
router.put('/team/:userId', authenticate, requireAdmin, asyncHandler(async (req, res) => {
  const { userId } = req.params;
  const { name, email, role, password } = req.body;

  // Can't demote yourself
  if (parseInt(userId) === req.userId && role && role !== 'admin') {
    return res.status(400).json({ error: 'Cannot change your own role' });
  }

  const updates = [];
  const values = [];

  if (name !== undefined) { updates.push('name = ?'); values.push(name); }
  if (email !== undefined) { updates.push('email = ?'); values.push(email.toLowerCase()); }
  if (role !== undefined) { updates.push('role = ?'); values.push(role); }
  
  if (password) {
    const hashedPassword = await bcrypt.hash(password, 10);
    updates.push('password = ?');
    values.push(hashedPassword);
  }

  if (updates.length === 0) {
    return res.status(400).json({ error: 'No fields to update' });
  }

  values.push(userId);
  await pool.execute(`UPDATE users SET ${updates.join(', ')} WHERE id = ?`, values);

  res.json({ message: 'User updated' });
}));

/**
 * DELETE /api/auth/team/:userId
 * Remove team member
 */
router.delete('/team/:userId', authenticate, requireAdmin, asyncHandler(async (req, res) => {
  const { userId } = req.params;

  // Can't delete yourself
  if (parseInt(userId) === req.userId) {
    return res.status(400).json({ error: 'Cannot delete yourself' });
  }

  await pool.execute('DELETE FROM users WHERE id = ?', [userId]);

  res.json({ message: 'User deleted' });
}));

module.exports = router;
