/**
 * Admin Routes - Single Tenant Version
 * Quiz management, questions, pages, logic
 * No account_id needed - client owns everything
 */

const express = require('express');
const router = express.Router();
const { v4: uuidv4 } = require('uuid');
const slugify = require('slugify');
const { pool, withTransaction } = require('../utils/db');
const { authenticate, requireEditor } = require('../middleware/auth');
const { asyncHandler } = require('../middleware/errorHandler');

// All routes require authentication
router.use(authenticate);

// ============================================
// Quizzes
// ============================================

/**
 * GET /api/admin/quizzes
 * List all quizzes
 */
router.get('/quizzes', asyncHandler(async (req, res) => {
  const [quizzes] = await pool.execute(`
    SELECT q.*, 
           (SELECT COUNT(*) FROM leads WHERE quiz_id = q.id) as lead_count,
           (SELECT COUNT(*) FROM questions WHERE quiz_id = q.id) as question_count
    FROM quizzes q
    ORDER BY q.created_at DESC
  `);

  res.json({ quizzes });
}));

/**
 * POST /api/admin/quizzes
 * Create new quiz
 */
router.post('/quizzes', requireEditor, asyncHandler(async (req, res) => {
  const { title, description } = req.body;

  if (!title) {
    return res.status(400).json({ error: 'Title required' });
  }

  const uuid = uuidv4();
  let slug = slugify(title, { lower: true, strict: true });

  // Ensure unique slug
  const [existing] = await pool.execute('SELECT id FROM quizzes WHERE slug = ?', [slug]);
  if (existing.length > 0) {
    slug = `${slug}-${Date.now().toString(36)}`;
  }

  const defaultDesign = {
    primaryColor: '#6366f1',
    secondaryColor: '#8b5cf6',
    backgroundColor: '#ffffff',
    textColor: '#1f2937',
    fontFamily: 'Inter'
  };

  const defaultSettings = {
    showProgressBar: true,
    allowBack: false,
    shuffleQuestions: false,
    shuffleOptions: false
  };

  await withTransaction(async (connection) => {
    // Create quiz
    const [result] = await connection.execute(
      `INSERT INTO quizzes (uuid, slug, title, description, design, settings) 
       VALUES (?, ?, ?, ?, ?, ?)`,
      [uuid, slug, title, description || '', JSON.stringify(defaultDesign), JSON.stringify(defaultSettings)]
    );

    const quizId = result.insertId;

    // Create default pages
    const pages = [
      { type: 'landing', title: 'Welcome', blocks: [] },
      { type: 'lead_capture', title: 'Get Your Results', blocks: [] },
      { type: 'result', title: 'Your Result', slug: 'result-1', blocks: [], is_default: 1 }
    ];

    for (const page of pages) {
      await connection.execute(
        `INSERT INTO pages (quiz_id, type, title, slug, blocks, is_default) VALUES (?, ?, ?, ?, ?, ?)`,
        [quizId, page.type, page.title, page.slug || null, JSON.stringify(page.blocks), page.is_default || 0]
      );
    }

    // Create default lead fields
    const fields = [
      { name: 'email', label: 'Email', type: 'email', required: 1, order: 0 },
      { name: 'name', label: 'Name', type: 'text', required: 0, order: 1 }
    ];

    for (const field of fields) {
      await connection.execute(
        `INSERT INTO lead_fields (quiz_id, field_name, field_label, field_type, required, order_index) 
         VALUES (?, ?, ?, ?, ?, ?)`,
        [quizId, field.name, field.label, field.type, field.required, field.order]
      );
    }

    res.status(201).json({
      quiz: { id: quizId, uuid, slug, title, description }
    });
  });
}));

/**
 * GET /api/admin/quizzes/:quizId
 * Get single quiz with all details
 */
router.get('/quizzes/:quizId', asyncHandler(async (req, res) => {
  const { quizId } = req.params;

  const [quizzes] = await pool.execute('SELECT * FROM quizzes WHERE id = ?', [quizId]);

  if (quizzes.length === 0) {
    return res.status(404).json({ error: 'Quiz not found' });
  }

  // Get questions with options
  const [questions] = await pool.execute(`
    SELECT q.*, 
           (SELECT JSON_ARRAYAGG(
              JSON_OBJECT('id', o.id, 'uuid', o.uuid, 'option_text', o.option_text, 
                          'option_image', o.option_image, 'icon', o.icon, 'score', o.score, 
                          'tags', o.tags, 'order_index', o.order_index)
           ) FROM question_options o WHERE o.question_id = q.id ORDER BY o.order_index) as options
    FROM questions q
    WHERE q.quiz_id = ?
    ORDER BY q.order_index
  `, [quizId]);

  // Get pages
  const [pages] = await pool.execute('SELECT * FROM pages WHERE quiz_id = ? ORDER BY type, id', [quizId]);

  // Get logic rules
  const [rules] = await pool.execute('SELECT * FROM logic_rules WHERE quiz_id = ? ORDER BY priority DESC', [quizId]);

  // Get lead fields
  const [fields] = await pool.execute('SELECT * FROM lead_fields WHERE quiz_id = ? ORDER BY order_index', [quizId]);

  // Get integrations
  const [integrations] = await pool.execute(`
    SELECT i.id, i.platform, i.name, i.enabled,
           qi.id as qi_id, qi.list_id, qi.form_id, qi.sheet_name, qi.tag_mapping, qi.field_mapping, qi.enabled as qi_enabled
    FROM integrations i
    LEFT JOIN quiz_integrations qi ON i.id = qi.integration_id AND qi.quiz_id = ?
    WHERE i.enabled = 1
  `, [quizId]);

  res.json({
    quiz: quizzes[0],
    questions: questions.map(q => ({ ...q, options: q.options || [] })),
    pages,
    rules,
    fields,
    integrations
  });
}));

/**
 * PUT /api/admin/quizzes/:quizId
 * Update quiz
 */
router.put('/quizzes/:quizId', requireEditor, asyncHandler(async (req, res) => {
  const { quizId } = req.params;
  const { title, description, slug, settings, design, seo, published, leadCapturePosition, leadCaptureAfterQuestion } = req.body;

  const updates = [];
  const values = [];

  if (title !== undefined) { updates.push('title = ?'); values.push(title); }
  if (description !== undefined) { updates.push('description = ?'); values.push(description); }
  if (settings !== undefined) { updates.push('settings = ?'); values.push(JSON.stringify(settings)); }
  if (design !== undefined) { updates.push('design = ?'); values.push(JSON.stringify(design)); }
  if (seo !== undefined) { updates.push('seo = ?'); values.push(JSON.stringify(seo)); }
  if (published !== undefined) { updates.push('published = ?'); values.push(published ? 1 : 0); }
  if (leadCapturePosition !== undefined) { updates.push('lead_capture_position = ?'); values.push(leadCapturePosition); }
  if (leadCaptureAfterQuestion !== undefined) { updates.push('lead_capture_after_question = ?'); values.push(leadCaptureAfterQuestion); }

  if (slug !== undefined) {
    const [existing] = await pool.execute('SELECT id FROM quizzes WHERE slug = ? AND id != ?', [slug, quizId]);
    if (existing.length > 0) {
      return res.status(409).json({ error: 'Slug already exists' });
    }
    updates.push('slug = ?');
    values.push(slugify(slug, { lower: true, strict: true }));
  }

  if (updates.length === 0) {
    return res.status(400).json({ error: 'No fields to update' });
  }

  values.push(quizId);
  await pool.execute(`UPDATE quizzes SET ${updates.join(', ')} WHERE id = ?`, values);

  res.json({ message: 'Quiz updated' });
}));

/**
 * DELETE /api/admin/quizzes/:quizId
 * Delete quiz (cascades to all related data)
 */
router.delete('/quizzes/:quizId', requireEditor, asyncHandler(async (req, res) => {
  const { quizId } = req.params;

  await pool.execute('DELETE FROM quizzes WHERE id = ?', [quizId]);

  res.json({ message: 'Quiz deleted' });
}));

/**
 * POST /api/admin/quizzes/:quizId/duplicate
 * Duplicate a quiz
 */
router.post('/quizzes/:quizId/duplicate', requireEditor, asyncHandler(async (req, res) => {
  const { quizId } = req.params;

  // Get original quiz
  const [quizzes] = await pool.execute('SELECT * FROM quizzes WHERE id = ?', [quizId]);
  if (quizzes.length === 0) {
    return res.status(404).json({ error: 'Quiz not found' });
  }

  const original = quizzes[0];
  const newUuid = uuidv4();
  const newSlug = `${original.slug}-copy-${Date.now().toString(36)}`;

  await withTransaction(async (connection) => {
    // Create new quiz
    const [result] = await connection.execute(
      `INSERT INTO quizzes (uuid, slug, title, description, settings, design, seo, lead_capture_position, lead_capture_after_question)
       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [newUuid, newSlug, `${original.title} (Copy)`, original.description, original.settings, original.design, original.seo, original.lead_capture_position, original.lead_capture_after_question]
    );

    const newQuizId = result.insertId;

    // Copy questions
    const [questions] = await connection.execute('SELECT * FROM questions WHERE quiz_id = ?', [quizId]);
    const questionMap = {};

    for (const q of questions) {
      const newQuestionUuid = uuidv4();
      const [qResult] = await connection.execute(
        `INSERT INTO questions (quiz_id, uuid, order_index, question_text, question_type, description, image_url, required, settings)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [newQuizId, newQuestionUuid, q.order_index, q.question_text, q.question_type, q.description, q.image_url, q.required, q.settings]
      );
      questionMap[q.id] = qResult.insertId;

      // Copy options
      const [options] = await connection.execute('SELECT * FROM question_options WHERE question_id = ?', [q.id]);
      for (const o of options) {
        await connection.execute(
          `INSERT INTO question_options (question_id, uuid, order_index, option_text, option_image, icon, score, tags)
           VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
          [qResult.insertId, uuidv4(), o.order_index, o.option_text, o.option_image, o.icon, o.score, o.tags]
        );
      }
    }

    // Copy pages
    const [pages] = await connection.execute('SELECT * FROM pages WHERE quiz_id = ?', [quizId]);
    for (const p of pages) {
      await connection.execute(
        `INSERT INTO pages (quiz_id, type, slug, title, blocks, settings, is_default) VALUES (?, ?, ?, ?, ?, ?, ?)`,
        [newQuizId, p.type, p.slug, p.title, p.blocks, p.settings, p.is_default]
      );
    }

    // Copy lead fields
    const [fields] = await connection.execute('SELECT * FROM lead_fields WHERE quiz_id = ?', [quizId]);
    for (const f of fields) {
      await connection.execute(
        `INSERT INTO lead_fields (quiz_id, field_name, field_label, field_type, placeholder, options, required, order_index, enabled)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [newQuizId, f.field_name, f.field_label, f.field_type, f.placeholder, f.options, f.required, f.order_index, f.enabled]
      );
    }

    res.status(201).json({
      quiz: { id: newQuizId, uuid: newUuid, slug: newSlug, title: `${original.title} (Copy)` }
    });
  });
}));

// ============================================
// Questions
// ============================================

/**
 * POST /api/admin/quizzes/:quizId/questions
 * Add question
 */
router.post('/quizzes/:quizId/questions', requireEditor, asyncHandler(async (req, res) => {
  const { quizId } = req.params;
  const { questionText, questionType = 'single', description, imageUrl, options = [] } = req.body;

  if (!questionText) {
    return res.status(400).json({ error: 'Question text required' });
  }

  await withTransaction(async (connection) => {
    // Get max order
    const [maxOrder] = await connection.execute(
      'SELECT MAX(order_index) as max_order FROM questions WHERE quiz_id = ?',
      [quizId]
    );
    const orderIndex = (maxOrder[0].max_order ?? -1) + 1;

    const uuid = uuidv4();
    const [result] = await connection.execute(
      `INSERT INTO questions (quiz_id, uuid, order_index, question_text, question_type, description, image_url)
       VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [quizId, uuid, orderIndex, questionText, questionType, description || null, imageUrl || null]
    );

    const questionId = result.insertId;

    // Add options
    for (let i = 0; i < options.length; i++) {
      const opt = options[i];
      await connection.execute(
        `INSERT INTO question_options (question_id, uuid, order_index, option_text, option_image, icon, score, tags)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
        [questionId, uuidv4(), i, opt.text || opt.option_text, opt.image || null, opt.icon || null, opt.score || 0, opt.tags ? JSON.stringify(opt.tags) : null]
      );
    }

    res.status(201).json({
      question: { id: questionId, uuid, order_index: orderIndex }
    });
  });
}));

/**
 * PUT /api/admin/questions/:questionId
 * Update question
 */
router.put('/questions/:questionId', requireEditor, asyncHandler(async (req, res) => {
  const { questionId } = req.params;
  const { questionText, questionType, description, imageUrl, required, settings } = req.body;

  const updates = [];
  const values = [];

  if (questionText !== undefined) { updates.push('question_text = ?'); values.push(questionText); }
  if (questionType !== undefined) { updates.push('question_type = ?'); values.push(questionType); }
  if (description !== undefined) { updates.push('description = ?'); values.push(description); }
  if (imageUrl !== undefined) { updates.push('image_url = ?'); values.push(imageUrl); }
  if (required !== undefined) { updates.push('required = ?'); values.push(required ? 1 : 0); }
  if (settings !== undefined) { updates.push('settings = ?'); values.push(JSON.stringify(settings)); }

  if (updates.length === 0) {
    return res.status(400).json({ error: 'No fields to update' });
  }

  values.push(questionId);
  await pool.execute(`UPDATE questions SET ${updates.join(', ')} WHERE id = ?`, values);

  res.json({ message: 'Question updated' });
}));

/**
 * DELETE /api/admin/questions/:questionId
 * Delete question
 */
router.delete('/questions/:questionId', requireEditor, asyncHandler(async (req, res) => {
  const { questionId } = req.params;

  // Get quiz_id and order for reordering
  const [questions] = await pool.execute('SELECT quiz_id, order_index FROM questions WHERE id = ?', [questionId]);
  if (questions.length === 0) {
    return res.status(404).json({ error: 'Question not found' });
  }

  const { quiz_id, order_index } = questions[0];

  await pool.execute('DELETE FROM questions WHERE id = ?', [questionId]);

  // Reorder remaining questions
  await pool.execute(
    'UPDATE questions SET order_index = order_index - 1 WHERE quiz_id = ? AND order_index > ?',
    [quiz_id, order_index]
  );

  res.json({ message: 'Question deleted' });
}));

/**
 * PUT /api/admin/quizzes/:quizId/questions/reorder
 * Reorder questions
 */
router.put('/quizzes/:quizId/questions/reorder', requireEditor, asyncHandler(async (req, res) => {
  const { quizId } = req.params;
  const { questionIds } = req.body;

  if (!Array.isArray(questionIds)) {
    return res.status(400).json({ error: 'questionIds array required' });
  }

  for (let i = 0; i < questionIds.length; i++) {
    await pool.execute(
      'UPDATE questions SET order_index = ? WHERE id = ? AND quiz_id = ?',
      [i, questionIds[i], quizId]
    );
  }

  res.json({ message: 'Questions reordered' });
}));

// ============================================
// Options
// ============================================

/**
 * POST /api/admin/questions/:questionId/options
 * Add option
 */
router.post('/questions/:questionId/options', requireEditor, asyncHandler(async (req, res) => {
  const { questionId } = req.params;
  const { optionText, optionImage, icon, score, tags } = req.body;

  if (!optionText) {
    return res.status(400).json({ error: 'Option text required' });
  }

  const [maxOrder] = await pool.execute(
    'SELECT MAX(order_index) as max_order FROM question_options WHERE question_id = ?',
    [questionId]
  );
  const orderIndex = (maxOrder[0].max_order ?? -1) + 1;

  const uuid = uuidv4();
  const [result] = await pool.execute(
    `INSERT INTO question_options (question_id, uuid, order_index, option_text, option_image, icon, score, tags)
     VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
    [questionId, uuid, orderIndex, optionText, optionImage || null, icon || null, score || 0, tags ? JSON.stringify(tags) : null]
  );

  res.status(201).json({
    option: { id: result.insertId, uuid }
  });
}));

/**
 * PUT /api/admin/options/:optionId
 * Update option
 */
router.put('/options/:optionId', requireEditor, asyncHandler(async (req, res) => {
  const { optionId } = req.params;
  const { optionText, optionImage, icon, score, tags } = req.body;

  const updates = [];
  const values = [];

  if (optionText !== undefined) { updates.push('option_text = ?'); values.push(optionText); }
  if (optionImage !== undefined) { updates.push('option_image = ?'); values.push(optionImage); }
  if (icon !== undefined) { updates.push('icon = ?'); values.push(icon); }
  if (score !== undefined) { updates.push('score = ?'); values.push(score); }
  if (tags !== undefined) { updates.push('tags = ?'); values.push(JSON.stringify(tags)); }

  if (updates.length === 0) {
    return res.status(400).json({ error: 'No fields to update' });
  }

  values.push(optionId);
  await pool.execute(`UPDATE question_options SET ${updates.join(', ')} WHERE id = ?`, values);

  res.json({ message: 'Option updated' });
}));

/**
 * DELETE /api/admin/options/:optionId
 * Delete option
 */
router.delete('/options/:optionId', requireEditor, asyncHandler(async (req, res) => {
  const { optionId } = req.params;
  await pool.execute('DELETE FROM question_options WHERE id = ?', [optionId]);
  res.json({ message: 'Option deleted' });
}));

// ============================================
// Pages
// ============================================

/**
 * PUT /api/admin/pages/:pageId
 * Update page (including result-specific Mailchimp/Kit tags)
 */
router.put('/pages/:pageId', requireEditor, asyncHandler(async (req, res) => {
  const { pageId } = req.params;
  const { title, slug, blocks, settings, isDefault, mailchimpTag, kitTag } = req.body;

  const updates = [];
  const values = [];

  if (title !== undefined) { updates.push('title = ?'); values.push(title); }
  if (slug !== undefined) { updates.push('slug = ?'); values.push(slug); }
  if (blocks !== undefined) { updates.push('blocks = ?'); values.push(JSON.stringify(blocks)); }
  if (settings !== undefined) { updates.push('settings = ?'); values.push(JSON.stringify(settings)); }
  if (isDefault !== undefined) { updates.push('is_default = ?'); values.push(isDefault ? 1 : 0); }
  if (mailchimpTag !== undefined) { updates.push('mailchimp_tag = ?'); values.push(mailchimpTag || null); }
  if (kitTag !== undefined) { updates.push('kit_tag = ?'); values.push(kitTag || null); }

  if (updates.length === 0) {
    return res.status(400).json({ error: 'No fields to update' });
  }

  values.push(pageId);
  await pool.execute(`UPDATE pages SET ${updates.join(', ')} WHERE id = ?`, values);

  res.json({ message: 'Page updated' });
}));

/**
 * POST /api/admin/quizzes/:quizId/pages
 * Add result page
 */
router.post('/quizzes/:quizId/pages', requireEditor, asyncHandler(async (req, res) => {
  const { quizId } = req.params;
  const { title, slug } = req.body;

  const [result] = await pool.execute(
    `INSERT INTO pages (quiz_id, type, title, slug, blocks, is_default) VALUES (?, 'result', ?, ?, '[]', 0)`,
    [quizId, title || 'New Result', slug || `result-${Date.now().toString(36)}`]
  );

  res.status(201).json({
    page: { id: result.insertId, type: 'result', title, slug }
  });
}));

/**
 * DELETE /api/admin/pages/:pageId
 * Delete result page
 */
router.delete('/pages/:pageId', requireEditor, asyncHandler(async (req, res) => {
  const { pageId } = req.params;

  // Can only delete non-default result pages
  const [pages] = await pool.execute('SELECT type, is_default FROM pages WHERE id = ?', [pageId]);
  if (pages.length === 0) {
    return res.status(404).json({ error: 'Page not found' });
  }

  if (pages[0].type !== 'result') {
    return res.status(400).json({ error: 'Can only delete result pages' });
  }

  if (pages[0].is_default) {
    return res.status(400).json({ error: 'Cannot delete default result page' });
  }

  await pool.execute('DELETE FROM pages WHERE id = ?', [pageId]);

  res.json({ message: 'Page deleted' });
}));

// ============================================
// Logic Rules
// ============================================

/**
 * POST /api/admin/quizzes/:quizId/rules
 * Add logic rule
 */
router.post('/quizzes/:quizId/rules', requireEditor, asyncHandler(async (req, res) => {
  const { quizId } = req.params;
  const { name, conditions, conditionMatch = 'all', actionType, actionValue, priority = 0 } = req.body;

  if (!conditions || !actionType || !actionValue) {
    return res.status(400).json({ error: 'Conditions, action type, and action value required' });
  }

  const uuid = uuidv4();
  const [result] = await pool.execute(
    `INSERT INTO logic_rules (quiz_id, uuid, name, conditions, condition_match, action_type, action_value, priority)
     VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
    [quizId, uuid, name || null, JSON.stringify(conditions), conditionMatch, actionType, JSON.stringify(actionValue), priority]
  );

  res.status(201).json({
    rule: { id: result.insertId, uuid }
  });
}));

/**
 * PUT /api/admin/rules/:ruleId
 * Update logic rule
 */
router.put('/rules/:ruleId', requireEditor, asyncHandler(async (req, res) => {
  const { ruleId } = req.params;
  const { name, conditions, conditionMatch, actionType, actionValue, priority, enabled } = req.body;

  const updates = [];
  const values = [];

  if (name !== undefined) { updates.push('name = ?'); values.push(name); }
  if (conditions !== undefined) { updates.push('conditions = ?'); values.push(JSON.stringify(conditions)); }
  if (conditionMatch !== undefined) { updates.push('condition_match = ?'); values.push(conditionMatch); }
  if (actionType !== undefined) { updates.push('action_type = ?'); values.push(actionType); }
  if (actionValue !== undefined) { updates.push('action_value = ?'); values.push(JSON.stringify(actionValue)); }
  if (priority !== undefined) { updates.push('priority = ?'); values.push(priority); }
  if (enabled !== undefined) { updates.push('enabled = ?'); values.push(enabled ? 1 : 0); }

  if (updates.length === 0) {
    return res.status(400).json({ error: 'No fields to update' });
  }

  values.push(ruleId);
  await pool.execute(`UPDATE logic_rules SET ${updates.join(', ')} WHERE id = ?`, values);

  res.json({ message: 'Rule updated' });
}));

/**
 * DELETE /api/admin/rules/:ruleId
 * Delete logic rule
 */
router.delete('/rules/:ruleId', requireEditor, asyncHandler(async (req, res) => {
  const { ruleId } = req.params;
  await pool.execute('DELETE FROM logic_rules WHERE id = ?', [ruleId]);
  res.json({ message: 'Rule deleted' });
}));

// ============================================
// Lead Fields
// ============================================

/**
 * PUT /api/admin/quizzes/:quizId/lead-fields
 * Update lead capture fields
 */
router.put('/quizzes/:quizId/lead-fields', requireEditor, asyncHandler(async (req, res) => {
  const { quizId } = req.params;
  const { fields } = req.body;

  if (!Array.isArray(fields)) {
    return res.status(400).json({ error: 'Fields array required' });
  }

  await withTransaction(async (connection) => {
    // Delete existing fields
    await connection.execute('DELETE FROM lead_fields WHERE quiz_id = ?', [quizId]);

    // Insert new fields
    for (let i = 0; i < fields.length; i++) {
      const f = fields[i];
      await connection.execute(
        `INSERT INTO lead_fields (quiz_id, field_name, field_label, field_type, placeholder, options, required, order_index, enabled)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [quizId, f.field_name, f.field_label, f.field_type || 'text', f.placeholder || null, f.options ? JSON.stringify(f.options) : null, f.required ? 1 : 0, i, f.enabled !== false ? 1 : 0]
      );
    }

    res.json({ message: 'Lead fields updated' });
  });
}));

module.exports = router;
