// Landing Page Designer Component
// This gets included in the main admin for quiz editing

function LandingPageDesigner({ quiz, onSave }) {
  const [design, setDesign] = React.useState(quiz?.design || {});
  const [saving, setSaving] = React.useState(false);
  const toast = useToast();

  const updateDesign = (key, value) => {
    setDesign(prev => ({ ...prev, [key]: value }));
  };

  const updateArrayItem = (key, index, value) => {
    const arr = [...(design[key] || [])];
    arr[index] = value;
    setDesign(prev => ({ ...prev, [key]: arr }));
  };

  const addArrayItem = (key, defaultValue = '') => {
    const arr = [...(design[key] || []), defaultValue];
    setDesign(prev => ({ ...prev, [key]: arr }));
  };

  const removeArrayItem = (key, index) => {
    const arr = (design[key] || []).filter((_, i) => i !== index);
    setDesign(prev => ({ ...prev, [key]: arr }));
  };

  async function handleSave() {
    setSaving(true);
    await onSave({ design });
    setSaving(false);
  }

  const defaultBenefits = [
    'Your personalized vulnerability score (with detailed breakdown)',
    'Specific gaps in your current protection that need immediate attention',
    'Custom action plan with steps you can implement today',
    'Industry insight - how you stack up against competitors',
    'Red flag warning signs to watch for in your business',
    'Professional recommendations tailored to your risk level'
  ];

  const defaultRiskItems = [
    'Your signature frameworks worth $25,000+ in development time',
    'Your client lists and proprietary relationship insights',
    'Your proven processes that took years to perfect',
    'Your competitive advantage in the marketplace',
    'Your business reputation and client trust'
  ];

  return (
    <div className="space-y-8">
      {/* Colors */}
      <div className="bg-white rounded-xl border p-6">
        <h3 className="font-bold text-lg mb-4">🎨 Brand Colors</h3>
        <div className="grid grid-cols-2 gap-4">
          <div>
            <label className="block text-sm font-medium mb-1">Primary Color (CTA buttons)</label>
            <div className="flex gap-2">
              <input type="color" value={design.primaryColor || '#f97316'} onChange={e => updateDesign('primaryColor', e.target.value)} className="w-12 h-10 rounded border cursor-pointer"/>
              <input type="text" value={design.primaryColor || '#f97316'} onChange={e => updateDesign('primaryColor', e.target.value)} className="flex-1 px-3 py-2 border rounded-lg font-mono text-sm"/>
            </div>
          </div>
          <div>
            <label className="block text-sm font-medium mb-1">Secondary Color (Hero background)</label>
            <div className="flex gap-2">
              <input type="color" value={design.secondaryColor || '#1e1b4b'} onChange={e => updateDesign('secondaryColor', e.target.value)} className="w-12 h-10 rounded border cursor-pointer"/>
              <input type="text" value={design.secondaryColor || '#1e1b4b'} onChange={e => updateDesign('secondaryColor', e.target.value)} className="flex-1 px-3 py-2 border rounded-lg font-mono text-sm"/>
            </div>
          </div>
        </div>
      </div>

      {/* Hero Section */}
      <div className="bg-white rounded-xl border p-6">
        <h3 className="font-bold text-lg mb-4">📢 Hero Section</h3>
        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium mb-1">Left Column Headline</label>
            <input type="text" value={design.landingHeadline || ''} onChange={e => updateDesign('landingHeadline', e.target.value)} placeholder='Take The FREE "Biz Risk" Quiz' className="w-full px-3 py-2 border rounded-lg"/>
          </div>
          <div>
            <label className="block text-sm font-medium mb-1">Left Column Subheadline</label>
            <textarea value={design.landingSubheadline || ''} onChange={e => updateDesign('landingSubheadline', e.target.value)} placeholder="Find out in 2 minutes. Discover your personalized results..." className="w-full px-3 py-2 border rounded-lg" rows={2}/>
          </div>
        </div>
      </div>

      {/* Lead Form */}
      <div className="bg-white rounded-xl border p-6">
        <h3 className="font-bold text-lg mb-4">📝 Lead Capture Form</h3>
        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium mb-1">Form Headline</label>
            <input type="text" value={design.formHeadline || ''} onChange={e => updateDesign('formHeadline', e.target.value)} placeholder="Enter Your Details to Begin!" className="w-full px-3 py-2 border rounded-lg"/>
          </div>
          <div>
            <label className="block text-sm font-medium mb-1">CTA Button Text</label>
            <input type="text" value={design.ctaText || ''} onChange={e => updateDesign('ctaText', e.target.value)} placeholder="Get My FREE Personalized Results" className="w-full px-3 py-2 border rounded-lg"/>
          </div>
        </div>
      </div>

      {/* Benefits Section */}
      <div className="bg-white rounded-xl border p-6">
        <h3 className="font-bold text-lg mb-4">✅ "What You'll Discover" Section</h3>
        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium mb-1">Section Headline</label>
            <input type="text" value={design.benefitsHeadline || ''} onChange={e => updateDesign('benefitsHeadline', e.target.value)} placeholder="What You'll Discover in 2 Minutes:" className="w-full px-3 py-2 border rounded-lg"/>
          </div>
          <div>
            <label className="block text-sm font-medium mb-1">Benefit Points</label>
            {(design.benefits || defaultBenefits).map((b, i) => (
              <div key={i} className="flex gap-2 mb-2">
                <input type="text" value={b} onChange={e => updateArrayItem('benefits', i, e.target.value)} className="flex-1 px-3 py-2 border rounded-lg text-sm"/>
                <button onClick={() => removeArrayItem('benefits', i)} className="text-red-500 px-2">✕</button>
              </div>
            ))}
            <button onClick={() => addArrayItem('benefits', '')} className="text-sm text-indigo-600 hover:underline">+ Add benefit</button>
          </div>
          <div>
            <label className="block text-sm font-medium mb-1">Bonus Text (optional)</label>
            <input type="text" value={design.bonusText || ''} onChange={e => updateDesign('bonusText', e.target.value)} placeholder="Get your detailed Copycat Risk Factor Report delivered instantly to your email." className="w-full px-3 py-2 border rounded-lg"/>
          </div>
        </div>
      </div>

      {/* Risk Section */}
      <div className="bg-white rounded-xl border p-6">
        <h3 className="font-bold text-lg mb-4">⚠️ "What's at Risk" Section</h3>
        <div className="space-y-4">
          <div className="flex items-center gap-2">
            <input type="checkbox" checked={design.showRiskSection !== false} onChange={e => updateDesign('showRiskSection', e.target.checked)} className="rounded"/>
            <label className="text-sm">Show this section</label>
          </div>
          <div>
            <label className="block text-sm font-medium mb-1">Section Headline</label>
            <input type="text" value={design.riskHeadline || ''} onChange={e => updateDesign('riskHeadline', e.target.value)} placeholder="What's Really at Risk?" className="w-full px-3 py-2 border rounded-lg"/>
          </div>
          <div>
            <label className="block text-sm font-medium mb-1">Risk Points</label>
            {(design.riskItems || defaultRiskItems).map((r, i) => (
              <div key={i} className="flex gap-2 mb-2">
                <input type="text" value={r} onChange={e => updateArrayItem('riskItems', i, e.target.value)} className="flex-1 px-3 py-2 border rounded-lg text-sm"/>
                <button onClick={() => removeArrayItem('riskItems', i)} className="text-red-500 px-2">✕</button>
              </div>
            ))}
            <button onClick={() => addArrayItem('riskItems', '')} className="text-sm text-indigo-600 hover:underline">+ Add risk point</button>
          </div>
          <div>
            <label className="block text-sm font-medium mb-1">CTA Text</label>
            <input type="text" value={design.riskCta || ''} onChange={e => updateDesign('riskCta', e.target.value)} placeholder="Most entrepreneurs have NO IDEA how exposed they really are." className="w-full px-3 py-2 border rounded-lg"/>
          </div>
        </div>
      </div>

      {/* Risk Levels Display */}
      <div className="bg-white rounded-xl border p-6">
        <h3 className="font-bold text-lg mb-4">🚦 Risk Level Indicators</h3>
        <div className="flex items-center gap-2 mb-4">
          <input type="checkbox" checked={design.showRiskLevels !== false} onChange={e => updateDesign('showRiskLevels', e.target.checked)} className="rounded"/>
          <label className="text-sm">Show High/Moderate/Low risk indicators on landing page</label>
        </div>
        <p className="text-sm text-gray-500">These are the colored dots showing "High Risk (65%)", "Moderate Risk", "Low Risk"</p>
      </div>

      {/* Bottom CTA */}
      <div className="bg-white rounded-xl border p-6">
        <h3 className="font-bold text-lg mb-4">👇 Bottom CTA</h3>
        <div>
          <label className="block text-sm font-medium mb-1">Bottom Button Text</label>
          <input type="text" value={design.bottomCtaText || ''} onChange={e => updateDesign('bottomCtaText', e.target.value)} placeholder="Get Your FREE Personalized Report" className="w-full px-3 py-2 border rounded-lg"/>
        </div>
      </div>

      <button onClick={handleSave} disabled={saving} className="w-full bg-indigo-600 text-white py-3 rounded-lg font-semibold hover:bg-indigo-700 disabled:opacity-50">
        {saving ? 'Saving...' : 'Save Landing Page Design'}
      </button>
    </div>
  );
}
